/***************************************************************************
  This is part of the evolver toolkit for exploring genetic progamming.
  Copyright (C) 1996 Benjamin Bennett and Yeasah G. Pell

  This library is free software; you can redistribute it and/or modify
  it under the terms of the GNU Library General Public License as
  published by the Free Software Foundation; either version 2 of the
  License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.

  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

  Contact information: Benjamin Bennett<fiji@limey.net> and Yeasah
    G. Pell<yeasah@wpi.edu>
  *************************************************************************/

#ifndef __GENERATION_H
#define __GENERATION_H

/************************************************************************
  Header file for the Generation class

    This class holds all the information about a particular generation
  of programs and contains functions to evaluate, mutate and report on
  the programs.

  **********************************************************************/

#include <stdlib.h>			// rand()

#include "generic.H"
#include "fitness.H"
#include "model.H"
#include "program.H"
#include "object.H"
#include "progfit.H"

// include the header file for the desired model
#if defined INT_MODEL
#include "optable.H"
#elif defined NET_MODEL
#include "optable_net1.H"
#endif // NET_MODEL

/*@ManMemo: This class performs the actual genetic evolution of the
  programs.  */
/*@Doc: This class holds all the information about a particular
  generation of programs and contains functions to evaluate, mutate
  and report on the programs. */
class GGeneration : public GObject
{
	/// The current generation (0 is first)
	int generation_num;
	/// The generated programs
	GArray<GProgram> programs;
	/// The evolving programs
	GArray<GProgram> new_programs;
	/// The list of models used to evaluate
	GArray<GModel> models;

	/// The minimum depth of a new tree	
	int stored_min_depth;
	/// The maximum depth of a new tree
	int stored_max_depth;

	/// The worst program we have seen
	double most_bad;

	// Helper functions for the breeding 
	/// Returns the size of the tree beneath the given node
	int GetSize(int type, GOperator *node);
	/// Finds a node that matches the given criteria
	int GetNode(int type, int *num, GOperator *node, GOperator **parent, 
				int *entry);
	/// Generates a new program
	GOperator *make_new_program(int min_depth, int max_depth, int cur_depth,
								int head_type, 
								type_enum ret_type = TYPE_INVALID);

	/// Perform the crossover operation on the given program indeces
	void Crossover(int prog_id1, int prog_id2);

	/// Mutate the given program
	void Mutate(int prog_id1);

	/// Select(copy) the given program
	void Select(int prog_id1);

	/// Evaluate the current generation with all of the models
	void EvaluateGeneration();

public:
	///
	GGeneration();
	///
	~GGeneration();

	/// Create a random set of programs and evaluate them
	void GenerateNewGeneration(int gen_size, int min_depth, int max_depth,
							   int head_type=-1);

	/// Breed the ``successful'' programs and evaluate the new ones
	void BreedNewGeneration(int num_progs, int prob_cross, int prob_mutate, 
							int prob_select);

	/// Return the ranking of the programs
	void Rank(GArray<GProgFit> &ranks);

	/// Return the current generation
	int GetGeneration();

	/// Add model to the model list to use in evaluation
	void AddModel(GModel *new_model);

	/// Get the specified program root
	GProgram *GetProgram(int prog_num);

	/// Get the number of programs that exist
	int GetNumProgs();

#ifdef DEBUG_GENERATION
	void TestCrossover(int prog_id1, int prog_id2);
	void TestSelect(int prog_id1);
#endif // DEBUG_GENERATION

};
#endif // __GENERATION_H
