/***************************************************************************
  This is part of the evolver toolkit for exploring genetic progamming.
  Copyright (C) 1996 Benjamin Bennett and Yeasah G. Pell

  This library is free software; you can redistribute it and/or modify
  it under the terms of the GNU Library General Public License as
  published by the Free Software Foundation; either version 2 of the
  License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.

  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

  Contact information: Benjamin Bennett<fiji@limey.net> and Yeasah
    G. Pell<yeasah@wpi.edu>
  *************************************************************************/

#ifndef __MODEL_H
#define __MODEL_H

/***************************************************************************
  GModel class - encapsulates definition of problem simulation and behaviour

  The GModel class has derived classes which define the behaviour
  **************************************************************************/

/* prerequisites */
#include "object.H"
#include "generic.H"
#include "fitness.H"
#include "operator.H"

// include the header file for the desired model
#if defined INT_MODEL
#include "type.H"
#include "optable.H"
#elif defined NET_MODEL
#include "type_net1.H"
#include "optable_net1.H"
#endif // NET_MODEL

/// Virtual class that defines the interface to derived models
/*@Doc: If a new model complies with this interface then it will be
  able to be evaluated by the existing code.  The other piece of code
  that needs to be added for a new model are the type definitions, and
  operator specifications.  Finally the models will have to be created
  and added in the users main function (or they can modify the
  existing one.) */
class GModel : public GObject
{
public:
	///
	GModel() {};
	///
	virtual ~GModel() {};

	/// Evaluate an operator (called by GProgram)
	/*Doc: This code is only model specific if you want to change the
      evaluation model.  The network model does an iterative
      evaluation of its nodes to allow for ``parallel'' evaluation.
      The integer model does a recursive evaluate.  The choice is up
      to the user and the model they wish to implement.  The iterative
      model is more general, but may use more memory.

	  The return value is ignored. */
	virtual int Evaluate(
		/// The head of the operator tree to be evaluated 
		GOperator *head) = 0;

	/// This actually does the dirty work in the evaluation
	/*Doc: The actual evaluation of the operators takes place within
	  this code.  The arguments, operator, and node are input.  The
	  return value must be the result of the evaluation, and be of the
	  correct sub-type. */
    virtual GType *Evaluate(
		/// The operation to evaluate
		oper_enum operation, 
		/// The arguments to the operator (these must be fully evaluated)
		GArray<GType> &params, 
		// The node that this operation is being executed in
		int node) = 0;

	/// return the next child to evaluate, or 0 when done
	/*@Doc: This allows for generality in the conditional operators.
	  Based upon the return value of the previous sub-tree, the index
	  to the previous sub-tree, and the operator this function must
	  return the index of the next child to evaluate, or zero when the
	  evalution is complete. */
	virtual int GetNext(
		/// The operation being evaluated
		oper_enum operation, 
		/// The last child that was evaluated or -1 if the first pass
		int last, 
		/// The returned value of the last child or NULL if the first pass
		GType *retval) = 0;

	/// returns the fitness for the last run
	/*@Doc: This must return the fitness of the program */
	virtual GFitness GetFitness() = 0;

	/// returns true if the tree should keep running
	virtual int Running() = 0;
};

#endif
