/***************************************************************************
  This is part of the evolver toolkit for exploring genetic progamming.
  Copyright (C) 1996 Benjamin Bennett and Yeasah G. Pell

  This library is free software; you can redistribute it and/or modify
  it under the terms of the GNU Library General Public License as
  published by the Free Software Foundation; either version 2 of the
  License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.

  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

  Contact information: Benjamin Bennett<fiji@limey.net> and Yeasah
    G. Pell<yeasah@wpi.edu>
  *************************************************************************/

#ifndef __NETWORK_H
#define __NETWORK_H

/***************************************************************************
  GNetwork - Functions to manipulate the network.

  The network and related functions.
 **************************************************************************/

/* prerequisites */
#include "generic.H"
#include "object.H"
#include "netnode.H"

class GNetNode;						// to prevent a circular declaration

/// Definition of the network and related functions
/*@Doc: This class encapsulates a complete network.  A network is
  defined as a collection of nodes with a defined source and sink. */
class GNetwork : public GObject
{
	/// The nodes contained in the network
	GArray<GNetNode> nodes;
	/// All of the pipes that are used in this network
	GArray<GPipe> pipes;
	/// The source of the flow
	int source;
	/// The destination for the flow
	int sink;
	/// The cached value for the maximum possible flow on the network
	int max_flow;

	int CalcFlow(int node, int *flow, int depth);
public:
	///
	GNetwork();
	///
	~GNetwork();
	/// Duplicates the network, nodes, pipes, messages, and queues
	GNetwork &operator=(GNetwork &input);

	/// Reset flows to zero, and clear the message queues
	void Reset();

	/// Increase the flow on the specified pipe and node by 1
    int IncreaseFlow(int node, int pipe);
	/// Decrease the flow on the specified pipe and node by 1
	int DecreaseFlow(int node, int pipe);
	/// Returns the flow on the specified input pipe and node
	int QueryInputFlow(int node, int pipe);
	/// Returns the flow on the specified output pipe and node
	int QueryOutputFlow(int node, int pipe);
	/// Returns the number of input pipes on node
	int QueryNumInPipes(int node);
	/// Returns the number of output pipes on node
	int QueryNumOutPipes(int node);
	/// Returns the capacity of the specified input pipe and node
	int QueryCapacityIn(int node, int pipe);
	/// Returns the capacity of the specified output pipe and node
	int QueryCapacityOut(int node, int pipe);
	/// Sends msg out along output pipe pipe with id msg_id for node
	int SendMessageOut(int node, int pipe, int msg_id, int msg);
	/// Sends msg out along input pipe pipe with id msg_id for node
	int SendMessageIn(int node, int pipe, int msg_id, int msg);
	/// Recieves output message of msg_id type on pipe pipe at node
	int RecieveMessageOut(int node, int pipe, int msg_id, int &msg);
	/// Recieves input message of msg_id type on pipe pipe at node
	int RecieveMessageIn(int node, int pipe, int msg_id, int &msg);

	/// Adds a new node, returns the node number of the new node
	int AddNode();

	/// Adds a new pipe from start to end with capacity capacity
	int AddPipe(int start, int end, int capacity);
	/// Adds the already defined pipe to the correct nodes
	int AddPipe(GPipe *pipe);

	/// Returns the current source
	int GetSource();
	/// Returns the current sink
	int GetSink();
	/// Sets the source
	int SetSource(int node);
	/// Sets the sink
	int SetSink(int node);

	/// Returns the number of nodes in the network
	int GetNumNodes();

	/// Print out the current network
	void Print();

	/// Calculate the maximum flow through the network
	int CalcMaxFlow();
};

#endif //__NETWORK_H
