/***************************************************************************
  This is part of the evolver toolkit for exploring genetic progamming.
  Copyright (C) 1996 Benjamin Bennett and Yeasah G. Pell

  This library is free software; you can redistribute it and/or modify
  it under the terms of the GNU Library General Public License as
  published by the Free Software Foundation; either version 2 of the
  License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.

  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

  Contact information: Benjamin Bennett<fiji@limey.net> and Yeasah
    G. Pell<yeasah@wpi.edu>
  *************************************************************************/

#ifndef __OPTABLE_H
#define __OPTABLE_H

/**************************************************************************
  GOperatorTable class - allows creation and querying of arbitrary
  operator types (without knowledge of derivation structure)
  ************************************************************************/

/* prerequisites */
#include "generic.H"
#include "type.H"

/* SYSTEM DEFINE - the total number of known operators */
#define NUM_OPERATORS 7

/* SYSTEM DEFINE - the maximum number of children any operator may have */
#define MAX_CHILDREN 2

/* operator enumeration */  
enum oper_enum
{
    OP_DATA_INT,
    OP_ADD_INT,
    OP_SUB_INT,
    OP_MUL_INT,
    OP_DIV_INT,
    OP_OUT_INT,
	OP_IN_INT
};

/* knows about all operators and types, and can create them based on enums */
class GOperatorTable : public GObject
{
	static int num_children[NUM_OPERATORS];
	static type_enum types[NUM_OPERATORS][MAX_CHILDREN+1];
	static char *names[NUM_OPERATORS];
	static int cond_type[NUM_OPERATORS];

public:
	/* returns number of known operators */
	int GetNumOperators()
	{
		return NUM_OPERATORS;
	}

	/* returns number of children, -1 if bad operator */
	int GetNumChildren(oper_enum op)
	{
		if(op >= 0 && op < NUM_OPERATORS)
			return num_children[op];
		else
			return -1;
	}

	/* returns name of operator */
	char *GetName(oper_enum op)
	{
		if(op >= 0 && op < NUM_OPERATORS)
			return names[op];
		else
			return NULL;
	}

	/* returns required child type */
	type_enum GetChildType(oper_enum op, int child_id)
	{
		if((op >= 0 && op < NUM_OPERATORS) &&
		   (child_id >= 0 && child_id < MAX_CHILDREN))
      
			return types[op][child_id+1];
		else
			return TYPE_INVALID;
	}

	/* returns the type enumeration value */
	type_enum GetType(oper_enum op)
	{
		if(op >= 0 && op < NUM_OPERATORS)
			return types[op][0];
		else
			return TYPE_INVALID;
	}

	/* returns 1 if a conditional 0 otherwise, -1 if bad operator */
	int CondType(oper_enum op)
	{
		if(op >= 0 && op < NUM_OPERATORS)
			return cond_type[op];
		else
			return -1;
	}
};

/* global table */
extern GOperatorTable optable;

#endif
