/***************************************************************************
  This is part of the evolver toolkit for exploring genetic progamming.
  Copyright (C) 1996 Benjamin Bennett and Yeasah G. Pell

  This library is free software; you can redistribute it and/or modify
  it under the terms of the GNU Library General Public License as
  published by the Free Software Foundation; either version 2 of the
  License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.

  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

  Contact information: Benjamin Bennett<fiji@limey.net> and Yeasah
    G. Pell<yeasah@wpi.edu>
  *************************************************************************/

/**************************************************************************
  GOperatorTable class - allows creation and querying of arbitrary
  operator types (without knowledge of derivation structure)
  ************************************************************************/

#include "optable_net1.H"

/* global declaration of operator table */
GOperatorTable optable;

/* array containing number of children per operator */
int GOperatorTable::num_children[NUM_OPERATORS] =
{
	// Environment interaction operations
    1,							// Increase flow: int (int pipe)
	1,							// Decrease flow: int (int pipe)
	1,							// Query flow in: int (int pipe)
	1,							// Query flow out: int (int pipe)
	0,							// Query num pipes: in int ()
	0,							// Query num pipes: out int ()
	1,							// Query capacity in: int (int pipe)
	1,							// Query capacity out: int (int pipe)
	2,							// Send message 1: int (int pipe, int msg)
	1,							// Recieve message 1: int (int pipe)
	2,							// Send message 2: int (int pipe, int msg)
	1,							// Recieve message 2: int (int pipe)
	2,							// Send message 1: int (int pipe, int msg)
	1,							// Recieve message 1: int (int pipe)
	2,							// Send message 2: int (int pipe, int msg)
	1,							// Recieve message 2: int (int pipe)
	0,							// Test for source node: bool ()
	0,							// Test for sink node: bool ()

	// Control operations
	3,							// If: int (bool cond, int if, int then)
	2,							// While: int (bool cond, int loop)

	// Logic operations
	2,							// Logical and: bool (bool a, bool b)
	2,							// Logical or: bool (bool a, bool b)
	1,							// Logical not: bool (bool a)
	2,							// Less than: bool (int a, int b)
	2,							// Equal to: bool (int a, int b)
	0,							// True

	// Variable operations
	0,							// Variable 1: int ()
	0,							// Variable 2: int ()
	2,							// Variable assignment int (int val, int dest)

	// Arithmetic operations
	0,							// The number 1: int ()
	2,							// Addition: int (int a, int b)
	2,							// Subtraction: int (int a, int b)
	2							// Multiplication: int (int a, int b)
};

/* array containing types of operators and children */
type_enum GOperatorTable::types[NUM_OPERATORS][MAX_CHILDREN+1] =
{
	/* return_type, child1_type, child2_type ... */
	// Environment interaction operations
    { TYPE_INT, TYPE_INT },		// Increase flow: int (int pipe)
	{ TYPE_INT, TYPE_INT },		// Decrease flow: int (int pipe)
	{ TYPE_INT, TYPE_INT },		// Query flow in: int (int pipe)
	{ TYPE_INT, TYPE_INT },		// Query flow out: int (int pipe)
	{ TYPE_INT },				// Query num pipes: in int ()
	{ TYPE_INT },				// Query num pipes: out int ()
	{ TYPE_INT, TYPE_INT },		// Query capacity in: int (int pipe)
	{ TYPE_INT, TYPE_INT },		// Query capacity out: int (int pipe)
	{ TYPE_INT, TYPE_INT, TYPE_INT }, // Send message 1
	{ TYPE_INT, TYPE_INT },		// Recieve message 1: int (int pipe)
	{ TYPE_INT, TYPE_INT, TYPE_INT }, // Send message 2
	{ TYPE_INT, TYPE_INT },		// Recieve message 2: int (int pipe)
	{ TYPE_INT, TYPE_INT, TYPE_INT }, // Send message 1
	{ TYPE_INT, TYPE_INT },		// Recieve message 1: int (int pipe)
	{ TYPE_INT, TYPE_INT, TYPE_INT }, // Send message 2
	{ TYPE_INT, TYPE_INT },		// Recieve message 2: int (int pipe)
	{ TYPE_BOOL },				// is source
	{ TYPE_BOOL },				// is sink

	// Control operations
	{ TYPE_INT, TYPE_BOOL, TYPE_INT, TYPE_INT }, // If
	{ TYPE_INT, TYPE_BOOL, TYPE_INT }, // While

	// Logic operations
	{ TYPE_BOOL, TYPE_BOOL, TYPE_BOOL},// Logical and: bool (bool a, bool b)
	{ TYPE_BOOL, TYPE_BOOL, TYPE_BOOL},	// Logical or: bool (bool a, bool b)
	{ TYPE_BOOL, TYPE_BOOL},	// Logical not: bool (bool a)
	{ TYPE_BOOL, TYPE_INT, TYPE_INT}, // Less than: bool (int a, int b)
	{ TYPE_BOOL, TYPE_INT, TYPE_INT}, // Equal to: bool (int a, int b)
	{ TYPE_BOOL },				// True

	// Variable operations
	{ TYPE_INT },				// Variable 1: int ()
	{ TYPE_INT },				// Variable 2: int ()
	{ TYPE_INT, TYPE_INT, TYPE_BOOL },	// Variable assignment int (int val, int dest)
	
	// Arithmetic operations
	{ TYPE_INT },				// The number 1: int ()
	{ TYPE_INT, TYPE_INT, TYPE_INT }, // Addition: int (int a, int b)
	{ TYPE_INT, TYPE_INT, TYPE_INT }, // Subtraction: int (int a, int b)
	{ TYPE_INT, TYPE_INT, TYPE_INT }  // Multiplication: int (int a, int b)
};

/* array of operator names */
char *GOperatorTable::names[NUM_OPERATORS] = 
{
	// Environment interaction operations
    "IncFlow",					// Increase flow: void (int pipe)
	"DecFlow",					// Decrease flow: void (int pipe)
	
	"FlowIn",					// Query flow in: int (int pipe)
	"FlowOut",					// Query flow out: int (int pipe)
	"NumPipesIn",				// Query num pipes: in int ()
	"NumPipesOut",				// Query num pipes: out int ()
	"CapacityIn",				// Query capacity in: int (int pipe)
	"CapacityOut",				// Query capacity out: int (int pipe)
	"SendOut(Msg1)",			// Send message 1: void (int pipe, int msg)
	"RecvOut(Msg1)",			// Recieve message 1: int (int pipe, int msg)
	"SendOut(Msg2)",			// Send message 2: void (int pipe, int msg)
	"RecvOut(Msg2)",			// Recieve message 2: int (int pipe, int msg)
	"SendIn(Msg1)",				// Send message 1: void (int pipe, int msg)
	"RecvIn(Msg1)",				// Recieve message 1: int (int pipe, int msg)
	"SendIn(Msg2)",				// Send message 2: void (int pipe, int msg)
	"RecvIn(Msg2)",				// Recieve message 2: int (int pipe, int msg)
	
	"IsSource?",				// Is source?
	"IsSink?",					// Is sink?

	// Control operations
	"If",						// If: void (bool cond, void if, void then)
	"While",					// While: void (bool cond, void loop)

	// Logic operations
	"And",						// Logical and: bool (bool a, bool b)
	"Or",						// Logical or: bool (bool a, bool b)
	"Not",						// Logical not: bool (bool a)
	"LessThan?",				// Less than: bool (int a, int b)
	"EqualTo?",					// Equal to: bool (int a, int b)
	"True",						// True

	// Variable operations
	"Var1",						// Variable 1: int ()
	"Var2",						// Variable 2: int ()
	"Assn",						// Variable assignment int (int val, int dest)
	
	// Arithmetic operations
	"One",						// The number 1: int ()
	"Add",						// Addition: int (int a, int b)
	"Sub",						// Unary negation: int (int a)
	"Mul"						// Multiplication: int (int a, int b)
};

/* array of conditional types */
int GOperatorTable::cond_type[NUM_OPERATORS] =
{
	// Environment interaction operations
    0,							// Increase flow: void (int pipe)
	0,							// Decrease flow: void (int pipe)
	0,							// Query flow in: int (int pipe)
	0,							// Query flow out: int (int pipe)
	0,							// Query num pipes: in int ()
	0,							// Query num pipes: out int ()
	0,							// Query capacity in: int (int pipe)
	0,							// Query capacity out: int (int pipe)
	0,							// Send message 1: void (int pipe, int msg)
	0,							// Recieve message 1: int (int pipe, int msg)
	0,							// Send message 2: void (int pipe, int msg)
	0,							// Recieve message 2: int (int pipe, int msg)
	0,							// Send message 1: void (int pipe, int msg)
	0,							// Recieve message 1: int (int pipe, int msg)
	0,							// Send message 2: void (int pipe, int msg)
	0,							// Recieve message 2: int (int pipe, int msg)
	0,							// is source
	0,							// is sink

	// Control operations
	1,							// If: void (bool cond, void if, void then)
	1,							// While: void (bool cond, void loop)

	// Logic operations
	0,							// Logical and: bool (bool a, bool b)
	0,							// Logical or: bool (bool a, bool b)
	0,							// Logical not: bool (bool a)
	0,							// Less than: bool (int a, int b)
	0,							// Equal to: bool (int a, int b)
	0,							// True

	// Variable operations
	0,							// Variable 1: int ()
	0,							// Variable 2: int ()
	0,							// Variable assignment int (int val, int dest)
	
	// Arithmetic operations
	0,							// The number 1: int ()
	0,							// Addition: int (int a, int b)
	0,							// Unary negation: int (int a)
	0							// Multiplication: int (int a, int b)
};
