/***************************************************************************
  This is part of the evolver toolkit for exploring genetic progamming.
  Copyright (C) 1996 Benjamin Bennett and Yeasah G. Pell

  This library is free software; you can redistribute it and/or modify
  it under the terms of the GNU Library General Public License as
  published by the Free Software Foundation; either version 2 of the
  License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.

  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

  Contact information: Benjamin Bennett<fiji@limey.net> and Yeasah
    G. Pell<yeasah@wpi.edu>
  *************************************************************************/

/***************************************************************************
  GPipe - Functions to manipulate pipes

 **************************************************************************/

#include "pipe.H"

GPipe::GPipe(GPipe &input)
{
	source = input.source;
	dest = input.dest;
	flow = input.flow;
	capacity = input.capacity;

	for(int i=0; i<2; i++)
	{
		in_queue[i] = input.in_queue[i];
		out_queue[i] = input.out_queue[i];
	}
}

GPipe::GPipe(int s, int d, int f, int c)
{
	source = s;
	dest = d;
	flow = f;
	capacity = c;
}

GPipe::~GPipe()
{
	for(int i=0; i<2; i++)
	{
		out_queue[i].DeleteQueue();
		in_queue[i].DeleteQueue();
	}
}

void GPipe::Reset()
{
	// get rid of messages
	for(int i=0; i<2; i++)
	{
		out_queue[i].DeleteQueue();
		in_queue[i].DeleteQueue();
	}

	// set flow to zero
	flow = 0;
}

GPipe& GPipe::operator=(GPipe &input)
{
	flow = input.flow;
	capacity = input.capacity;
	source = input.source;
	dest = input.dest;

	for(int i=0; i<2; i++)
	{
		in_queue[i] = input.in_queue[i];
		out_queue[i] = input.out_queue[i];
	}

	return *this;
}

int GPipe::ChangeFlow(int change)
{
	flow += change;
	if (flow < 0)
		flow = 0;
	if (flow > capacity)
		flow = capacity;

	return 1;
}

int GPipe::SetCapacity(int cap)
{
	capacity = cap;

	return 1;
}

int GPipe::QueryFlow()
{
	return flow;
}

int GPipe::QueryCapacity()
{
	return capacity;
}

int GPipe::SendMessageOut(int msg_id, int msg)
{
	GMessage *tmp;

	if (msg_id > 1 || msg_id < 0)
	{
		cerr << "GPipe::SendMessageOut - invalid message id: " 
			 << msg_id << endl;
		exit(1);
	}

	tmp = new GMessage(msg_id, msg);

	out_queue[msg_id].Add(tmp);

	return 1;
}

int GPipe::SendMessageIn(int msg_id, int msg)
{
	GMessage *tmp;

	if (msg_id > 1 || msg_id < 0)
	{
		cerr << "GPipe::SendMessageOut - invalid message id: " 
			 << msg_id << endl;
		exit(1);
	}
	
	tmp = new GMessage(msg_id, msg);
	in_queue[msg_id].Add(tmp);

	return 1;
}

int GPipe::RecieveMessageOut(int msg_id, int &msg)
{
	GMessage *tmp;

	if (msg_id > 1 || msg_id < 0)
	{
		cerr << "GPipe::RecieveMessageOut - invalid message id: " 
			 << msg_id << endl;
		exit(1);
	}

	msg_id -= 1;
	tmp = in_queue[msg_id].Pop();
	if (tmp != NULL)
	{
		msg_id = tmp->message_id;
		msg = tmp->message;
		return 0;
	}

	return 1;
}

int GPipe::RecieveMessageIn(int msg_id, int &msg)
{
	GMessage *tmp;

	if (msg_id > 1 || msg_id < 0)
	{
		cerr << "GPipe::RecieveMessageIn - invalid message id: " 
			 << msg_id << endl;
		exit(1);
	}

	tmp = out_queue[msg_id].Pop();
	if (tmp != NULL)
	{
		msg_id = tmp->message_id;
		msg = tmp->message;
		return 0;
	}

	return 1;
}

void GPipe::Print()
{

	cout << "\tPipe from " << source << " to " << dest << " flow " << flow 
		 << " of " << capacity << endl;
	cout << "\t\tCurrently carrying " << in_queue[0].GetSize() << ", "
		 << in_queue[1].GetSize() << " messages in and " 
		 << out_queue[0].GetSize() << ", " << out_queue[0].GetSize() 
			 << " out" << endl;
}
