/***************************************************************************
  This is part of the evolver toolkit for exploring genetic progamming.
  Copyright (C) 1996 Benjamin Bennett and Yeasah G. Pell

  This library is free software; you can redistribute it and/or modify
  it under the terms of the GNU Library General Public License as
  published by the Free Software Foundation; either version 2 of the
  License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.

  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

  Contact information: Benjamin Bennett<fiji@limey.net> and Yeasah
    G. Pell<yeasah@wpi.edu>
  *************************************************************************/

/* test suite for the generic classes */
#include "generic.H"
#include <iostream.h>

const int CHUNK_SIZE = 16;
const int NUM_TESTS = 256;

int numcmp(int *a, int *b)
{
	if (*a > *b)
		return 1;
	if (*a == *b)
		return 0;

    return -1;
}

int main(int argc, char* argv[])
{
	int success = 1;

	/* test resizing */
	{
		GIntArray a;

		cout << "Testing resizing functions" << endl << endl;
		cout << "Initial size: " << a.GetSize() << endl;
		a.SetSize(NUM_TESTS);
		cout << "Size after setting to " << NUM_TESTS << ": " << a.GetSize() 
			<< endl;
		if (a.GetSize() != NUM_TESTS)
			success = 0;

		a.SetSize(0);
		cout << "Size after setting to 0: " << a.GetSize() << endl << endl;
        if (a.GetSize() != 0)
			success = 0;
	}
	if (!success)
    {
		cerr << "ERROR: Resizing tests failed" << endl;
		exit(1);
    }

	/* test the dynamic array stuff */
	{
		GIntArray a;
		GIntArray b;
		int i;
    
		i = (NUM_TESTS / CHUNK_SIZE) - 1;

		cout << "Starting sequential test, should increase size " <<
			i << " times" << endl;
    
		for (i = 0; i < NUM_TESTS; i++)
			a[i] = i;
    
		for (i = 0; i < NUM_TESTS; i++)
		{
			if (i % (CHUNK_SIZE/2) == 0)
				cout << endl;
			cout << i << ':' << a[i] << "\t";
			if (i != a[i])
				success = 0;
		}
		cout << endl;
    
#if HOLES
		cout << "\nTesting non-linear assignment\n";
		b[0]=0;
		b[50]=50;
		b[2136]=2136;
		b[10]=10;
		cout << "Confirm the value pairs are identical:" << endl;
		cout << "0:" << b[0] << " 10:" << b[10] << " 50:" << b[50] 
			 << " 2136:" << b[2136] << endl; 
    	if ((0 != b[0]) && (50 != b[50]) && (b[2136] != 2136) && (10 != b[10]))
			success = 0;
#endif
	}
	if (!success)
    {
		cerr << "ERROR: Assignment tests failed" << endl;
		exit(1);
    }


	/* now test the stack-like stuff */
	{
		int i,j;
		GIntArray a;
		GIntArray b;
    
		cout << "\n\nStarting Append test, numbers should add to 255" << endl;

		for (i = 0; i < NUM_TESTS; i++)
			a.Append(i);
    
		for (i = 0; i < NUM_TESTS; i++)
		{
			if (i % (CHUNK_SIZE/2) == 0)
				cout << endl;
			j = a.Pop();
			cout << i << ':' << j << "\t";
			if (i + j != NUM_TESTS - 1)
				success = 0;
		}
		cout << '\n';
		if (a.GetSize() == 0)
			cout << "Stack empty, good" << endl;
		else
		{
			cout << "Oops, stack not drained, " << a.GetSize() 
				<< " still on stack" << endl;
			success = 0;
		}
	}
	if (!success)
    {
		cerr << "ERROR: Stack tests failed" << endl;
		exit(1);
    }

	/* now test Last() stuff */
	{
		int i;
		GIntArray a;
    
		cout << "\n\nTesting Last()" << endl;

		for (i = 0; i < NUM_TESTS; i++)
		{
			a.Append(i);
			if ((i % (CHUNK_SIZE/2)) == 0)
			{
				cout << "Last: " << i << " = " << a.Last() << endl;
				cout << "Setting " << i << " (using last) to 0" << endl;
				a.Last() = 0;
			}
		}

		cout << "\nThere should be a 0 every " << (CHUNK_SIZE/2) << " numbers" 
			 << endl;
		for (i = 0; i < NUM_TESTS; i++)
		{
			if (i % (CHUNK_SIZE/2) == 0)
				cout << endl;
			cout << i << ':' << a[i] << "\t";
			if (i % (CHUNK_SIZE/2) == 0) 
			{
				if (0 != a[i])
				{
					cout << "0 - " << a[i] << endl;
					success = 0;
				}
			}
			else if (a[i] != i)
			{
				success = 0;
				cout << i << " - " << a[i] << endl;
			}
		}
		cout << endl;
	}
	if (!success)
    {
		cerr << "ERROR: Last tests failed" << endl;
		exit(1);
    }


	/* test = stuff for arrays (not array elements) */
	{
		GIntArray a;
		GIntArray b;
		int i;
    
		i = (NUM_TESTS / CHUNK_SIZE) - 1;

		cout << endl << endl <<
			"Starting array copy test, confirm pairs are identical" << endl;
    
		for (i = 0; i < NUM_TESTS; i++)
			a[i] = i;

		b = a;			// perform copy
		a.SetSize(0);		// kill a
   
		for (i = 0; i < NUM_TESTS; i++)
			a[i] = NUM_TESTS - i;	// and reset it with other stuff

		for (i = 0; i < NUM_TESTS; i++)
		{
			if (i % (CHUNK_SIZE/2) == 0)
				cout << endl;
			cout << i << ':' << b[i] << "\t";
			if (b[i] != i)
				success = 0;
		}
		cout << endl;
	}
	if (!success)
    {
		cerr << "ERROR: Last tests failed" << endl;
		exit(1);
    }

	/* test sort stuff  */
	{
		GIntArray a;
		int i;
    
		cout << endl << endl << "Starting sort test" << endl;
    
		for (i = 0; i < NUM_TESTS; i++)
			a[i] = NUM_TESTS - i - 1;

		cout << "Before sort" << endl;

		for (i = 0; i < NUM_TESTS; i++)
		{
			if (i % (CHUNK_SIZE/2) == 0)
				cout << endl;
			cout << i << ':' << a[i] << "\t";
		}

		a.Sort((int (*)(const void *, const void *))numcmp);
    
		cout << endl << endl << "After sort" << endl;

		for (i = 0; i < NUM_TESTS; i++)
		{
			if (i % (CHUNK_SIZE/2) == 0)
				cout << endl;
			cout << i << ':' << a[i] << "\t";
			if (i != a[i])
				success = 0;
		}
		cout << endl;
	}
	if (!success)
    {
		cerr << "ERROR: Sort test failed" << endl;
		exit(1);
    }

	cerr << "All tests succeeded" << endl;
  
	return 0;
}
