/***************************************************************************
  This is part of the evolver toolkit for exploring genetic progamming.
  Copyright (C) 1996 Benjamin Bennett and Yeasah G. Pell

  This library is free software; you can redistribute it and/or modify
  it under the terms of the GNU Library General Public License as
  published by the Free Software Foundation; either version 2 of the
  License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.

  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

  Contact information: Benjamin Bennett<fiji@limey.net> and Yeasah
    G. Pell<yeasah@wpi.edu>
  *************************************************************************/

/***************************************************************************
  GTestModel class - test definition
 **************************************************************************/

#include <limits.h>
#include "test_model.H"

GTestModel::GTestModel(int inp, int max_iters)
{
	max_iterations = max_iters;
	input = inp;
	iteration = 0;
	out = 0;
}

/* returns the fitness values for a target of final evaluation = 50 */
GFitness GTestModel::GetFitness()
{
	GFitness fit;
	int desired;				// desired result

	/* set the desired result */
	desired = input * input * input + input * input - input;

	/* reset number of iterations  */
	/* (assuming we do a GetFitness() at the end of a run */
	iteration = 0;

	/* see if there was an exact hit */
	if(out == desired) fit.hits = 1;
	else fit.hits = 0;
	
	/* now calculate the distance from the desired value */
	fit.fitness = (float)out-desired;
	if (fit.fitness < 0.0) fit.fitness = fit.fitness * -1.0;

#ifdef DEBUG
	cerr << "GTestModel::GetFitness(): hits=" << fit.hits << " fitness=" 
		 << fit.fitness << endl;
#endif

	return fit;
}


/* evaluates a given operator */
GType *GTestModel::Evaluate(oper_enum operation, GArray<GType> &params, 
							int node)
{
	int a[MAX_CHILDREN];
	int retval = -1;
	
    /* display diagnostics */
#ifdef DEBUG
	cerr << "GTestModel::Evaluate(" << optable.GetName(operation);
	for(int i=0; i<params.GetSize(); i++)
		cerr << ", " << (*(int *)(params[i]->GetData()));
	cerr << ")";
#endif
	
	out = 0;					// reset out

	/* check for the right number of children */
	if(optable.GetNumChildren(operation) == params.GetSize())
	{
		/* read the parameters */
		for(int i=0; i<optable.GetNumChildren(operation); i++)
			a[i] = *(int *)(params[i]->GetData());
	
			/* do the appropriate operation */
			switch(operation)
			{
			case OP_DATA_INT:
	            retval = 1;
	            break;
	  
	        case OP_ADD_INT:
	            retval = a[0] + a[1];
	            break;
	  
	        case OP_MUL_INT:
	            retval = a[0] * a[1];
	            break;
	  
	        case OP_SUB_INT:
	            retval = a[0] - a[1];
	            break;
	  
	        case OP_DIV_INT:
	            if(a[1] != 0.0)
	                retval = a[0] / a[1];
            	else
	                retval = INT_MAX;
	            break;
	  
	        case OP_OUT_INT:
	            retval = out = a[0];
	            break;

	        case OP_IN_INT:
	            retval = input;
	            break;
			}
	}


#ifdef DEBUG
	cerr << " = " << retval << endl;
#endif
  
	/* return value */
	return new GIntType(retval);
}

/* evaluates this operator (evaluating children in the process) */
GType *GTestModel::Evaluate(GOperator *head, int tmp)
{
	int i;
	GArray<GType> params;
	GType *retval;
  
	tmp = 0;
	/* if we shouldn't keep running, or we aren't valid, just
	   return NULL and let the model figure it out */
	if(!head->Valid() || !Running())
		return NULL;

#ifdef DEBUG
	cerr << "Evaluating Operator Type: " << 
		optable.GetName(head->GetOperatorType()) << endl;
	for(i=0; i< head->children.GetSize(); i++)
		cerr << optable.GetName(head->children[i]->GetOperatorType()) << ", ";
	cerr << endl;
#endif // DEBUG

	// If a conditional operator
	if (optable.CondType(head->GetOperatorType()))
	{
		retval = NULL;
		i = 0;
		// ask model which child to eval next
		// which_next should free retval and set it when it returns 0
		while( (i = GetNext(head->GetOperatorType(), i, retval)) )
		{
			retval = Evaluate(head->children[i], 1);
			if(retval == NULL) return NULL;
		}
	}
	else
	{
		/* evaluate children, build list */
		for(i=0; i<head->children.GetSize(); i++)
		{
			retval = Evaluate(head->children[i], 1);
			if(retval == NULL) return NULL;
			params[i] = retval;
		}
		
		/* do the actual operation */
		retval = Evaluate(head->GetOperatorType(), params, 1);
	}

	return retval;
}

/* evaluates this operator (evaluating children in the process) */
int GTestModel::Evaluate(GOperator *head)
{
	delete Evaluate(head, 1);
	return 1;
}

/* return the next child to evaluate, or 0 when done */
int GTestModel::GetNext(oper_enum operation, int last, GType *value)
{
	int retval;

	delete value;				// free up the last value
	if (retval == 0)			// this is the last pass
		value = new GIntType(1);

	return retval;
}

/* decision function for length of runtime for a certain program */
int GTestModel::Running()
{
	iteration++;
	if(iteration > max_iterations)
		return 0;
	else
		return 1;
}
