/***************************************************************************
  This is part of the evolver toolkit for exploring genetic progamming.
  Copyright (C) 1996 Benjamin Bennett and Yeasah G. Pell

  This library is free software; you can redistribute it and/or modify
  it under the terms of the GNU Library General Public License as
  published by the Free Software Foundation; either version 2 of the
  License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.

  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

  Contact information: Benjamin Bennett<fiji@limey.net> and Yeasah
    G. Pell<yeasah@wpi.edu>
  *************************************************************************/

#ifndef __type_h
#define __type_h

/***************************************************************************
  Type class - encapsulates data passed between operators
  *************************************************************************/

/* prerequisites */
#include "generic.H"

/* SYSTEM DEFINE - total number of known types */
#define NUM_TYPES 2
    
/// The valid types defined for the network model
enum type_enum
{
	/// An invalid type, this needs to be defined because it is referenced elsewhere in the code
	TYPE_INVALID=-1,
	/// The integer type
	TYPE_INT=0,
	/// The boolean type
	TYPE_BOOL=1
};  

/// Operator return type class
/*@Doc: This class provides the interface to the user defined types
  required for differnt models.  If a model needs a new type then it
  must be a descendant of {\tt GType} and it must define at least the
  functionality of {\tt GType}. */
class GType : public GObject
{
public:
	/// Returns the type enumeration value
	virtual type_enum GetType() = 0;

	/// Gives a pointer to the data
	virtual void *GetData() = 0;

	/// Sets data given a pointer
	virtual void SetData(void *adata) = 0;
};

/// The integer class 
/*@Doc: This class provides the same functionality as the C++ integer
  type it is built upon.  There are no special restrictions upon its
  use. */
class GIntType : public GType
{
	/// Data payload
	int data;

public:
	/// Constructor, requires the data value as input
	GIntType(int adata)
	{
		data = adata;
	}

	/// Returns the type of this object
	virtual type_enum GetType()
	{
		return TYPE_INT;
	}

	/// Returns the data value that this object holds
	virtual void *GetData()
	{
		return &data;
	}

	/// Sets the data value of the object
	virtual void SetData(void *adata)
	{
		data = *((int *)adata);
	}
};

/// Boolean class, for logical operations
/*@Doc:
  This class is used by operators which take a boolean value as input.  
  This class holds true and false, with one representing true and zero
  representing false.  Note that this is compatible with C++'s concept 
  of truth but is a little more restrictive. {\tt SetData()} and the 
  constructor both convert all non-zero values to one automatically.
 */
class GBoolType : public GType
{
	/// Data payload
	int data;

public:
	/// Constructor, takes an integer as input with false = 0, else true
	GBoolType(int adata)
	{
		data = adata?1:0;		// 0 if 0, 1 otherwise
	}

	/// Gets the object type
	virtual type_enum GetType()
	{
		return TYPE_BOOL;
	}

	/// Gets the value of the data
	virtual void *GetData()
	{
		return &data;
	}

	/// Sets the data value
	virtual void SetData(void *adata)
	{
		data = *((int *)adata)?1:0; // 0 if 0, 1 otherwise
	}
};

#endif
